<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Latest_Posts_Widget extends WP_Widget {

    // Widget setup.
    public function __construct() {
        $widget_ops = array(
            'classname'   => 'latest-posts-widget',
            'description' => esc_html__( 'Display the latest posts with title, featured image, author name, and comment count.', 'pointify' ),
        );
        parent::__construct( 'latest_posts_widget', esc_html__( 'Latest Posts', 'pointify' ), $widget_ops );
    }

    // Output the content of the widget.
    public function widget( $args, $instance ) {
        echo $args['before_widget'];

        if ( ! empty( $instance['title'] ) ) {
            echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'] ) . $args['after_title'];
        }

        $query_args = array(
            'post_type'      => 'post',
            'posts_per_page' => $instance['number'],
        );

        $latest_posts = new WP_Query( $query_args );

        if ( $latest_posts->have_posts() ) :
            while ( $latest_posts->have_posts() ) : $latest_posts->the_post(); ?>

                <div class="latest-post">
                    <?php if ( has_post_thumbnail() ) : ?>
                        <div class="post-thumbnail">
                            <a href="<?php the_permalink(); ?>" title="<?php the_title(); ?>">
                                <?php the_post_thumbnail( 'thumbnail' ); ?>
							</a>
							<span class="post-category"><?php pointify_get_the_category(); ?></span>
                        </div>
                    <?php endif; ?>

                    <div class="post-content">
                        <h4><a href="<?php the_permalink(); ?>" title="<?php the_title(); ?>"><?php the_title(); ?></a></h4>
						<?php
							if ( has_excerpt() ) {
							?>
								<p class="sidebar_description"><?php echo wp_trim_words(get_the_excerpt(), 40, ''); ?></p>
							<?php
							} else { 
								?>
									<p class="sidebar_description"><?php echo substr( get_the_content(), 0, 40 ) . ' ' . __( '...', 'pointify' ); ?></p>
								<?php
							}
						?>
						
                        <p class="author-name"><?php esc_html_e( 'By', 'pointify' ); ?> <?php the_author(); ?></p>
                        <p class="comment-count"><?php comments_number( '0', '1', '%' ); ?></p>
                    </div>
                </div>

            <?php endwhile;
            wp_reset_postdata();
        endif;

        echo $args['after_widget'];
    }

    // Widget settings form.
    public function form( $instance ) {
        $title  = ! empty( $instance['title'] ) ? $instance['title'] : esc_html__( 'Latest Posts', 'pointify' );
        $number = ! empty( $instance['number'] ) ? $instance['number'] : 5;
        ?>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'pointify' ); ?></label>
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>">
        </p>

        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"><?php esc_html_e( 'Number of posts to display:', 'pointify' ); ?></label>
            <input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'number' ) ); ?>" type="number" value="<?php echo esc_attr( $number ); ?>" min="1">
        </p>

        <?php
    }

    // Save widget settings.
    public function update( $new_instance, $old_instance ) {
        $instance          = array();
        $instance['title'] = ( ! empty( $new_instance['title'] ) ) ? sanitize_text_field( $new_instance['title'] ) : esc_html__( 'Latest Posts', 'pointify' );
        $instance['number'] = ( ! empty( $new_instance['number'] ) ) ? absint( $new_instance['number'] ) : 5;

        return $instance;
    }
}

// Register the widget.
function register_latest_posts_widget() {
    register_widget( 'Latest_Posts_Widget' );
}
add_action( 'widgets_init', 'register_latest_posts_widget' );